function [out,fmin,parout] = fdpg(Ffun,F_grad_conj,Gfun,Gfun_prox,Afun,Atfun,lambda,starty,par)
% FDPG employs the fast dual proximal gradient method for solving the problem min{f(x) + lambda* g(A*x)}
%
% Underlying assumptions:
% All functions are convex
% f is proper closed and strongly convex 
% g is proper closed and proximable
% A is a linear transformation
% lambda is a positive scalar
% ====================================================
% Usage:
% out               = FDPG(Ffun,F_grad_conj,Gfun,Gfun_prox,Afun,Atfun,lambda,starty,[par])
% [out,fmin]        = FDPG(Ffun,F_grad_conj,Gfun,Gfun_prox,Afun,Atfun,lambda,starty,[par])
% [out,fmin,parout] = FDPG(Ffun,F_grad_conj,Gfun,Gfun_prox,Afun,Atfun,lambda,starty,[par])
% ====================================================
% Input:
% Ffun        - function handle for the function f
% F_grad_conj - function handle which calculates for input vector w: argmax_x { <x,w>  - f(x) }
% Gfun        - function handle for the function g
% Gfun_prox   - function handle for the proximal mapping of g
% Afun        - function handle for the linear transformation of A
% Atfun       - function handle for the adjoint of the linear transformation of A
% lambda      - positive scalar penalty for the function g
% starty      - starting vector of dual variables (resides in the image space of A)
% par         - struct which contains different values required for the operation of FDPG
% Fields of par:
%       max_iter         - maximal number of iterations [default: 1000]
%       eco_flag         - true if economic version (without calculating objective function
%                          values) should run, otherwise false [default: false]
%       print_flag       - true if internal printing should take place, otherwise false [default: true]
%       Lstart           - initial value for the Lipschitz constant (scalar) [default: 1]
%       const_flag       - true if constant step size should be used, otherwise false [default: false]
%       regret_flag      - true if the Lipschitz constant should be divided by eta in the next iteration [default: false]
%       real_valued_flag - true if g is assumed to be a real-valued function [default: false]
%                          when the flag is set to true, only improved function values are printed
%       feas_check_flag  - true if feasibly violation should be printed [default: true]
%                          If real_valued_flag is true, then feas_check_flag is set to false
%       eta              - the multiplicative constant used when regretting or backtracking [default: 2]
%       eps              - stopping criteria tolerance (the method stops when the
%                          norm of the difference between consecutive iterates is < eps) [default: 1e-5]
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    FeasVal   - vecror of all feasibly violation values (when feas_check_flag is true)
%    LValVec   - vector of all used Lipschitz constants (when par.const_flag is false) 
%    dualVec   - dual optimal vector

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

if ((nargin < 8) || ( isa (Gfun_prox,'function_handle')~= 1) || ( isa (Afun,'function_handle')~= 1) || ...
        ( isa (Atfun,'function_handle')~= 1) || ( isa (F_grad_conj,'function_handle')~= 1)  )
    error ('usage: fdpg(Ffun,F_grad_conj,Gfun,Gfun_prox,Afun,Atfun,lambda,starty,[par])') ;
end


%setting default values

if (nargin < 9)
    par = struct() ;
end

if ( isempty(Ffun) || isempty(Gfun) || ~isa(Ffun,'function_handle') || ~isa(Gfun,'function_handle') )
    if (nargout > 1)
        error('When FFun or Gfun is not a function handle usage: out = fdpg(Ffun,F_grad_conj,Gfun,Prox_fun,Afun,Atfun,lambda,starty,[par])')
    else
        if ( ~isfield (par,'eco_flag') || (~par.eco_flag))
            fprintf('Ffun or Gfun is not a function handle, changing to economical mode\n') ;
            par.eco_flag = true ;
            par.print_flag = false ;
            par.const_flag = true ;
            par.feas_check_flag = false ;
        end
    end
end
par = initialize_par_for_fdpg(par) ;

%constant values
Lstart = par.Lstart ;
const_flag = par.const_flag ;
regret_flag = par.regret_flag ;
max_iter = par.max_iter ;
print_flag = par.print_flag ;
eta = par.eta ;
eps = par.eps ;
eco_flag = par.eco_flag ;
real_valued_flag = par.real_valued_flag ;
feas_check_flag = par.feas_check_flag ;

Feps = 1e-10 ; %small epsilon for comparing newFvalue and quad_aprrox

%starting values
iternu=0; %iteration number
prevy = starty ;
done = false ;
prevw = starty ;
prevt = 1 ;
maxval = F_grad_conj(Atfun(prevw)) ; % argmax{ <x, At*prevy - f(x)| = starting value for u
if (~eco_flag)
    Fvalue = Ffun(maxval) ;
    Gvalue = Gfun (Afun(maxval)) ;
    best_value = Fvalue+lambda*Gvalue ; % best value so far
end
best_vector = maxval ; %the vector giving the best value
best_dual = starty ;

if (nargout == 3)
    parout = struct ;
    parout.iterNum = 0 ;
    parout.funValVec = [] ;
    if (feas_check_flag)
        parout.feasVal = [] ;
    end
    if (~const_flag)
        parout.LValVec = [] ;
    end
end

%starting the L according to const_flag and no_regret
if (const_flag)
    L = Lstart ;
else
    if (regret_flag)
        L = Lstart * eta ; % in the first iteration it will be divided by eta
    else
        L = Lstart ;
    end
end

%First printing
if(print_flag)
    fprintf('*********************\n');
    fprintf('fdpg\n') ;
    fprintf('*********************\n')
    fprintf('#iter       \tfun. val.   \t')
    if (feas_check_flag)
        fprintf('feas. viol. \t')
    end
    if (~const_flag)
        fprintf('L val.');
    end
    fprintf('\n')
end

%the iterations
while( (iternu < max_iter) && (~done))
    iternu = iternu + 1;
    
    if ((~const_flag) && (regret_flag))
        L = L / eta ;
    end
    
    newu = maxval ; %F_grad_conj(Atfun(prevw))
    newv = Gfun_prox(Afun(newu) - L * prevw, lambda*L) ;
    newy = prevw - 1/L * (Afun(newu) - newv) ;
    newt = 1 + sqrt(1 + 4 * prevt^2)/2 ;
    neww = newy + (prevt-1)/newt*(newy - prevy) ;
    maxval = F_grad_conj(Atfun(neww)) ;
    
    if (~const_flag)
        %search for appropiate L
        val_prevw = trace(newu'* Atfun(prevw)) - Ffun(newu) ; 
        val_neww = trace(maxval'*Atfun(neww)) - Ffun(maxval) ; 
        while (val_neww > val_prevw + sum(sum(Afun(newu).*(neww - prevw))) + L/2 * norm(neww-prevw,'fro')^2 + Feps)
            L = L * eta ;
            newv = Gfun_prox(Afun(newu) - L * prevw, lambda*L) ;
            newy = prevw - 1/L * (Afun(newu) - newv) ;
            newt = 1 + sqrt(1 + 4 * prevt^2)/2 ;
            neww = newy + (prevt-1)/newt*(newy - prevy) ;
            maxval = F_grad_conj(Atfun(neww)) ;
            val_neww = trace(maxval'*Atfun(neww)) - Ffun(maxval) ; 
        end
    end
    
    
    if (~eco_flag)
        Fvalue = Ffun(newu) ;
        Gvalue = Gfun (Afun(newu)) ;
        res_value = Fvalue+lambda*Gvalue ;
        if (~real_valued_flag || (res_value < best_value))
            best_value = res_value ;
            best_vector = newu ;
            best_dual = neww ;

            if ((print_flag || (nargout == 3)) && feas_check_flag)
                feas_value = norm(Afun(newu)-Gfun_prox(Afun(newu),1e-8),'fro') ;
            end
            if (print_flag)
                fprintf('%6d \t    %12f \t',iternu,best_value);
                if (feas_check_flag)
                    fprintf('%14g \t',feas_value) ;
                end
                if (~const_flag)
                    fprintf('%12f',L) ;
                end
                fprintf('\n') ;
            end
            
        end
        if (nargout == 3)
            %updating parout
            parout.funValVec = [ parout.funValVec ; best_value] ;
            if (feas_check_flag)
                parout.feasVal = [parout.feasVal ; feas_value ] ;
            end
            if (~const_flag)
                parout.LValVec = [parout.LValVec ; L] ;
            end
        end
    end
    
    if (iternu > 1)
        if (norm(newu - prevu,'fro') < eps)
            done = true ;
        end
    end
    
    prevu = newu ;
    prevy = newy ;
    prevt = newt ;
    prevw = neww ;
end

if (done && print_flag)
    fprintf('Stopping because the norm of the difference between consecutive iterates is too small\n')
end

%moving to the best value we found

if (eco_flag)
    out = newu ;
else
    out = best_vector ;
end

if ((nargout >1) || (print_flag))
    if (~eco_flag)
        fmin = best_value ;
    else
        fmin = Ffun(out)+lambda*Gfun(Afun(out)) ;
    end
end

if (nargout == 3)
    parout.iterNum = iternu ;
    parout.dualVec = best_dual ;
end

if (print_flag)
    fprintf('----------------------------------\n') ;
    fprintf('Optimal value = %15f \n',fmin) ;
end

end



